/**
* \file: libtestbedapp.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: application-framework
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "libtestbedapp.h"
#include "libtestbedapp_intern.h"

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdarg.h>

#include "application_state_machine.h"
#include "dbus_interface.h"

//----------------------------------------- private attributes ----------------------------------------
static tba_loglevel_t log_level;
static log_event_cb_t log_cb;
static void *cb_user_data;

//----------------------------------------- public member ---------------------------------------------
tba_error_code_t libtestbedapp_init(bool service_available,const char *client_busname, tba_loglevel_t arg_log_level,
		unsigned int shutdown_timeout, const tba_callbacks_t *callbacks, void *callback_user_data)
{
	log_level=arg_log_level;
	cb_user_data=callback_user_data;

	log_debug("libtestbedapp - Initializing testbed application.");
	log_debug("Busname: %s, service available: %s", client_busname, service_available ? "true" : "false");

	if (callbacks!=NULL)
		log_cb=callbacks->log_event_cb;

	asm_init(service_available,callbacks, callback_user_data);
	return dbus_iface_init(client_busname, shutdown_timeout);
}

tba_app_state_t libtestbedapp_get_state(void)
{
	return asm_get_state();
}

tba_app_state_t libtestbedapp_get_previous_state(void)
{
	return asm_get_previous_state();
}

const char *libtestbedapp_get_error_message(void)
{
	return asm_get_error_message();
}

tba_error_code_t libtestbedapp_get_error_code(void)
{
	return asm_get_error_code();
}

unsigned int libtestbedapp_get_surface_id(void)
{
	return asm_get_surface_id();
}

tba_loglevel_t libtestbedapp_get_loglevel(void)
{
	return log_level;
}

void libtestbedapp_set_loglevel(tba_loglevel_t new_log_level)
{
	log_level=new_log_level;
}

void libtestbedapp_send_to_background(void)
{
	log_debug("libtestbedapp - Application requests sending itself to background.");
	asm_send_to_background();
}

void libtestbedapp_send_to_foreground(void)
{
	log_debug("libtestbedapp - Application requests sending itself to foreground.");
	asm_send_to_foreground();
}

void libtestbedapp_service_lost(void)
{
	log_debug("libtestbedapp - Application signals a lost service.");
	asm_service_lost();
}

void libtestbedapp_service_available(void)
{
	log_debug("libtestbedapp - Application signals service availability.");
	asm_service_available();
}

void libtestbedapp_deinit(void)
{
	log_debug("libtestbedapp - Deinitializing testbed application.");
	dbus_iface_deinit();
	asm_deinit();
}

void libtestbedapp_enter_shutdown_state(int request_id)
{
	log_debug("libtestbedapp - Application requests entering shutdown state (request_id: %d)", request_id);
	asm_set_state(TBA_STATE_SHUTDOWN);
	if (request_id>=0)
		dbus_iface_answer_pending_lc_request((unsigned int)request_id);
}

void libtestbedapp_enter_active_state(tba_app_state_t state, int request_id)
{
	assert(state==TBA_STATE_FOREGROUND || state==TBA_STATE_BACKGROUND || state==TBA_STATE_NOSERVICE);
	log_debug("libtestbedapp - Application requests active state (request_id: %d, state: %d)", request_id, state);
	asm_set_state(state);
	if (request_id>=0)
		dbus_iface_answer_pending_lc_request((unsigned int)request_id);
}

void libtestbedapp_log(const char *prefix, int prefix_len, tba_loglevel_t msg_log_level, const char *message, ...)
{
	char whole_msg[LOG_MESSAGE_MAX+prefix_len];
	char *msg_ptr;
	va_list args;

	if (msg_log_level < log_level || log_cb == NULL)
		return;

	strcpy(whole_msg,prefix);
	msg_ptr=whole_msg+prefix_len;

	memset(&args,0,sizeof(va_list));
	va_start (args, message);
	vsnprintf(msg_ptr, LOG_MESSAGE_MAX, message, args);
	va_end(args);
	log_cb(msg_log_level,whole_msg, cb_user_data);
}
